/**
* \file: device_list.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <stdbool.h>

#include "model/model_internal.h"
#include "model/device_list.h"
#include "model/partition_list.h"


//the head element of the linked device list
device_t *list_head=NULL;

error_code_t device_list_init(void)
{
	if (list_head!=NULL)
		device_list_deinit();

	list_head=malloc(sizeof(device_t));
	if (list_head!=NULL)
	{
		list_head->state=_UNKNOWN_DEVICE_STATE;
		list_head->next_device=NULL;
		list_head->previous_device=NULL;
		list_head->id=NULL;
	}
	else
		return RESULT_NORESOURCE;

	return RESULT_OK;
}

void device_list_deinit(void)
{
	if (list_head == NULL) return;

	device_t *device=list_head->next_device;

	//cleanly empty the list
	while (device != NULL)
	{
		device_t *next_dev=device->next_device;
		device_free(device);
		device=next_dev;
	}

	free(list_head);
	list_head=NULL;
}

device_t *device_list_add_device(const char* device_id, device_metadata_t *metadata,
		devicehandler_vtable_t *vtable)
{
	device_t *device, *first_device;

	device = device_create_new(device_id, metadata, vtable);
	if (device!=NULL)
	{
		//put in the new device between head and the first device
		first_device = list_head->next_device;

		//update forward links
		list_head->next_device=device;
		device->next_device=first_device;

		//update backwards links
		device->previous_device=list_head;
		if (first_device!=NULL)
			first_device->previous_device=device;
	}

	return device;
}

void device_list_delete_device(device_t *device)
{
	device_t *dev_before_device, *dev_after_device;

	dev_before_device=device->previous_device;
	dev_after_device=device->next_device;

	//update the forward link
	dev_before_device->next_device=dev_after_device;

	//update the backward link
	if (dev_after_device!=NULL)
		dev_after_device->previous_device=dev_before_device;

	//delete the device
	device_free(device);
}

device_t* device_list_first_element(device_list_iterator_t *iterator)
{
	if (iterator != NULL)
		(*iterator)=(device_list_iterator_t)list_head->next_device;

	return list_head->next_device;
}

device_t* device_list_next_element(device_list_iterator_t *iterator)
{
	device_t *current_element=(device_t *)(*iterator);
	if (current_element==NULL)
		return NULL;
	(*iterator)=(device_list_iterator_t)current_element->next_device;

	return current_element->next_device;
}

device_t* device_list_find_device(const char *device_id)
{
	device_t *device;
	device=list_head->next_device;

	while (device != NULL)
	{
		if (strcmp(device->id,device_id)==0)
			return device;
		device=device->next_device;
	}

	//not found
	return NULL;
}

device_t* device_list_find_device_by_mountpoint(const char *mount_point)
{
	partition_t *partition;
	partition=device_list_find_partition_by_mountpoint(mount_point);
	if (partition==NULL)
		return NULL;

	return partition->device;
}
